﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using System.Threading;
using RevisionAnalyser.Controls;
using SharpSvn;
using RevisionAnalyser.Global;
using System.Collections.ObjectModel;

namespace RevisionAnalyser.Projects.RecoderProject
{
    public partial class AddRevisionForm : LoadableForm
    {
        private List<long> _analysed;
        private double _percentage = 10.0;
        private RecoderProject _project = (RecoderProject)CurrentProject.Instance.Project;
        private SvnClient _svn;

        public AddRevisionForm(List<long> analysed)
        {
            _analysed = analysed;
            InitializeComponent();
        }

        private void AddRevisionForm_Load(object sender, EventArgs e)
        {
            AddRevisionForm_SizeChanged(sender, e);
            uxLimitField.SelectedIndex = 2;
        }

        public List<Revision> GetSelectedRevisions()
        {
            List<Revision> revisions = new List<Revision>();
            foreach (ListViewItem item in uxRevisionList.CheckedItems)
            {
                try
                {
                    DateTime time;
                    long id = long.Parse(item.SubItems[1].Text);
                    string author = item.SubItems[2].Text;
                    DateTime.TryParse(item.SubItems[3].Text, out time);
                    string logMessage = item.SubItems[4].Text;
                    revisions.Add(new SvnRevision(id, author, time, logMessage));
                }
                catch
                {
                }
            }
            return revisions;
        }

        private void InitDateFields()
        {
            ShowLoadScreen("Fetching first/last revision dates from SVN...");
            try
            {
                _svn = _project.GetSvnClient();
                // Get last revision date
                SvnInfoEventArgs infoArgs;
                _svn.GetInfo(new SvnUriTarget(_project.SvnUrl), out infoArgs);
                uxToField.Value = uxToField.MaxDate = uxFromField.MaxDate = infoArgs.LastChangeTime.Date;
                // Get first revision date
                Collection<SvnLogEventArgs> logItems;
                try
                {
                    _svn.GetLog(new Uri(_project.SvnUrl), new SvnLogArgs(new SvnRevisionRange(0, 0)), out logItems);
                    uxFromField.Value = uxFromField.MinDate = uxToField.MinDate = logItems[0].Time.Date;
                }
                catch
                {
                    uxFromField.Value = uxToField.Value;
                }
            }
            catch (Exception ex)
            {
                TopMostMessageBox.Show(String.Format("SVN connection error.\n\n{0}", ex.Message), "SVN connection error", MessageBoxButtons.OK, MessageBoxIcon.Error);
            }
            finally
            {
                HideLoadScreen();
            }
        }

        private void Search(SvnLogArgs args)
        {
            ShowLoadScreen("Searching for revisions...");
            uxSearchButton.Enabled = false;
            uxAdvancedSearchButton.Enabled = false;
            try
            {
                if (!uxLimitField.Text.Equals("None"))
                {
                    args.Limit = Int32.Parse(uxLimitField.Text);
                }
                Collection<SvnLogEventArgs> logItems;
                _svn.GetLog(new Uri(_project.SvnUrl), args, out logItems);
                uxRevisionList.Items.Clear();
                foreach (SvnLogEventArgs item in logItems)
                {
                    DateTime localTime = item.Time.ToLocalTime();
                    foreach (SvnRevisionRange range in args.Ranges)
                    {
                        if (range.StartRevision.Time.CompareTo(localTime) < 0 && range.EndRevision.Time.CompareTo(localTime) > 0)
                        {
                            ListViewItem listItem = new ListViewItem();
                            listItem.SubItems.Add(item.Revision.ToString());
                            listItem.SubItems.Add(item.Author);
                            listItem.SubItems.Add(localTime.ToString());
                            listItem.SubItems.Add(item.LogMessage);
                            if (_analysed.Contains(item.Revision))
                            {
                                listItem.BackColor = Constants.COLOR_GREEN;
                                listItem.ForeColor = Color.Gray;
                            }
                            uxRevisionList.Items.Add(listItem);
                            break;
                        }
                    }
                }
                uxSelectAllButton.Enabled = uxSelectNoneButton.Enabled = uxSelectPercentageButton.Enabled = uxRevisionList.Items.Count > 0;
            }
            catch (Exception ex)
            {
                TopMostMessageBox.Show(String.Format("SVN connection error.\n\n{0}", ex.Message), "SVN connection error", MessageBoxButtons.OK, MessageBoxIcon.Error);
            }
            finally
            {
                uxSearchButton.Enabled = true;
                uxAdvancedSearchButton.Enabled = true;
                AddRevisionForm_SizeChanged(null, null);
                HideLoadScreen();
            }
        }

        private void Search()
        {
            if (uxToField.Value.Date.CompareTo(uxFromField.Value) < 0)
            {
                MessageBox.Show("Invalid date range.", "Invalid date range", MessageBoxButtons.OK, MessageBoxIcon.Warning);
                return;
            }
            SvnLogArgs args = new SvnLogArgs();
            args.Range = new SvnRevisionRange(Functions.GetSimpleUtcDate(uxFromField.Value), Functions.GetSimpleUtcDate(uxToField.Value.AddDays(1)));
            Search(args);
        }

        private void AdvancedSearch()
        {
            MultiDateForm frmMultiDate = new MultiDateForm(uxFromField.Value, uxToField.MaxDate, uxLimitField.SelectedIndex);
            if (frmMultiDate.ShowDialog() == DialogResult.OK)
            {
                uxLimitField.SelectedIndex = frmMultiDate.GetLimitIndex();
                SvnLogArgs args = new SvnLogArgs();
                SortedDictionary<DateTime, DateTime> dates = frmMultiDate.GetDates();
                foreach (KeyValuePair<DateTime, DateTime> date in dates)
                {
                    args.Ranges.Add(new SvnRevisionRange(Functions.GetSimpleUtcDate(date.Key), Functions.GetSimpleUtcDate(date.Value.AddDays(1))));
                }
                Search(args);
            }
        }

        private void SelectAll()
        {
            foreach (ListViewItem item in uxRevisionList.Items)
            {
                item.Checked = true;
            }
        }

        private void SelectPercentage(double percentage)
        {
            _percentage = percentage;
            if (percentage >= 0 && percentage <= 100)
            {
                if (percentage == 0)
                {
                    SelectNone();
                }
                else if (percentage == 100)
                {
                    SelectAll();
                }
                else
                {
                    SelectNone();
                    double stepSize = uxRevisionList.Items.Count / (percentage / 100 * uxRevisionList.Items.Count);
                    if (stepSize == 0)
                    {
                        stepSize = 1;
                    }
                    for (double d = 0; d < uxRevisionList.Items.Count; d += stepSize)
                    {
                        int i = (int)Math.Floor(d);
                        uxRevisionList.Items[i].Checked = true;
                    }
                }
            }
        }

        private void SelectPercentage()
        {
            SelectPercentageForm frmSelectPercentage = new SelectPercentageForm(_percentage);
            if (frmSelectPercentage.ShowDialog() == DialogResult.OK)
            {
                SelectPercentage(frmSelectPercentage.GetPercentage());
            }
        }

        private void SelectNone()
        {
            foreach (ListViewItem item in uxRevisionList.Items)
            {
                item.Checked = false;
            }
        }

        private void AddRevisionForm_Shown(object sender, EventArgs e)
        {
            InitDateFields();
        }

        private void uxSearchButton_Click(object sender, EventArgs e)
        {
            Search();
        }

        private void uxAdvancedSearchButton_Click(object sender, EventArgs e)
        {
            AdvancedSearch();
        }

        private void AddRevisionForm_FormClosing(object sender, FormClosingEventArgs e)
        {
            if (_svn != null)
            {
                _svn.Dispose();
            }
        }

        private void uxSelectAllButton_Click(object sender, EventArgs e)
        {
            SelectAll();
        }

        private void uxSelectPercentageButton_Click(object sender, EventArgs e)
        {
            SelectPercentage();
        }

        private void uxSelectNoneButton_Click(object sender, EventArgs e)
        {
            SelectNone();
        }

        private void uxRevisionList_ItemChecked(object sender, ItemCheckedEventArgs e)
        {
            if (e.Item.ForeColor == Color.Gray)
            {
                e.Item.Checked = false;
            }
            if (e.Item.Checked)
            {
                e.Item.BackColor = Constants.COLOR_DARK_GREEN;
            }
            else
            {
                if (e.Item.ForeColor == Color.Gray)
                {
                    e.Item.BackColor = Constants.COLOR_GREEN;
                }
                else
                {
                    e.Item.BackColor = Color.Transparent;
                }
            }
            uxAddButton.Enabled = uxRevisionList.CheckedItems.Count > 0;
        }

        private void uxRevisionList_ItemSelectionChanged(object sender, ListViewItemSelectionChangedEventArgs e)
        {
            if (e.IsSelected)
            {
                e.Item.Checked = !e.Item.Checked;
            }
            e.Item.Selected = false;
        }

        private void uxAddButton_Click(object sender, EventArgs e)
        {
            DialogResult = DialogResult.OK;
            Close();
        }

        private void AddRevisionForm_SizeChanged(object sender, EventArgs e)
        {
            uxMessageHeader.Width = Width - 425;
        }
    }
}
